{ TestPasStr unit

  Copyright (C) 2014 by Alexander Savinykh hinstance@yandex.ru

  This library is free software; you can redistribute it and/or modify it under the terms of the
  GNU Library General Public License as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version with the following modification:

  As a special exception, the copyright holders of this library give you permission to link this
  library with independent modules to produce an executable, regardless of the license terms of
  these independent modules,and to copy and distribute the resulting executable under terms of
  your choice, provided that you also meet, for each linked independent module, the terms and
  conditions of the license of that module. An independent module is a module which is not derived
  from or based on this library. If you modify this library, you may extend this exception to your
  version of the library, but you are not obligated to do so. If you do not wish to do so, delete
  this exception statement from your version.

  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
  the GNU Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License along with this
  library; if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
}

unit TestPasStrU;

{$Include PasStr.inc}

interface

uses
  types,
  sysutils,
  strutils,
  Classes,
  TestFramework,
  TimeManager,
  EpikTimer,
  StringBuilderU,
  StringLinkedListU,
  StringUtilU;

const
  DefaultTestingStringArrayLength = 20000;
  DefaultTestingStringLengthA = 10;
  DefaultTestingStringLengthB = 400;
  RandomStringSymbols = 'abcdefdhijklmnop1234567890';
  NonRandomSymbol = '?';

procedure RegisterAllTests;

type

  TNestedRoutine = procedure is nested;

  { TTestPasStr }

  TTestPasStr = class(TTestCase)
  protected
    FPerformanceLog: TStrings;
  public
    property PerfLog: TStrings read FPerformanceLog write FPerformanceLog;
  public
    procedure SetUpOnce; override;
  public
    procedure RunPerf(a: TNestedRoutine; const s: string);
  public
    procedure SavePerfLog;
  public
    procedure TearDownOnce; override;
  end;

  { TTestStringBuilder }

  TTestStringBuilder = class(TTestCase)
  published
    procedure TestEmptyString;
  published
    procedure TestCreateDestroy;
  published
    procedure TestSingleItem;
  published
    procedure TestTwoItems;
  published
    procedure TestTwoItemsA;
  published
    procedure TestWithClean;
  published
    procedure TestRus;
  published
    procedure TestCleanBeforeFree;
  published
    procedure TestSingleChar;
  published
    procedure TestSingleChars;
  end;

  { TTestStringLinkedList }

  TTestStringLinkedList = class(TTestPasStr)
  published
    procedure TestEmpty;
  published
    procedure TestSingle;
  published
    procedure TestGenerateRandomString;
  public
    procedure CheckForEach(const a: TStringDynArray; aList: TStringLinkedList);
  public
    procedure CheckListToString(
      const aExpected: TStringDynArray;
      aList: TStringLinkedList;
      const aSeparator : string
    );
  published
    procedure TestMultiple;
  end;

  { TTestStringUtil }

  TTestStringUtil = class(TTestPasStr)
  published
    procedure TestOpenStringArrayToDynArraySimple;
  published
    procedure TestOpenStringArrayToDynArrayEmpty;
  published
    procedure TestSplitStringSimple;
  public
    {Test SplitString function with specified separator}
    procedure TestSplitStringWithSep(aSeparator: Char);
  published
    procedure TestSplitStringMultiple;
  published
    procedure TestStringArrayContainsNormal;
  published
    procedure TestStringCopyABSimple;
  end;

implementation

procedure RegisterAllTests;
begin
  RegisterTest(TTestStringBuilder.Suite);
  RegisterTest(TTestStringLinkedList.Suite);
  RegisterTest(TTestStringUtil.Suite);
end;

function GenerateRandomString(a, b: Integer): string;
const
  s = RandomStringSymbols;
var
  n, i: Integer;
begin
  n := a + Random(b - a + 1);
  SetLength(result, n);
  for i := 0 to n - 1 do
    result[i + 1] := s[Random(Length(s)) + 1];
end;

function GenerateTestingArray(a, b, n: Integer): TStringDynArray;
var
  i: Integer;
begin
  SetLength(result, n);
  for i := 0 to n - 1 do
    result[i] := GenerateRandomString(a, b);
end;

{ TTestPasStr }

procedure TTestPasStr.SetUpOnce;
begin
  inherited SetUpOnce;
  PerfLog := TStringList.Create;
end;

procedure TTestPasStr.RunPerf(a: TNestedRoutine; const s: string);
var
  t: TimerData;
begin
  gTimer.Clear(t);
  gTimer.Start(t);
  a();
  gTimer.Stop(t);
  PerfLog.Add(s + '=' + gTimer.ElapsedDHMS(t));
end;

procedure TTestPasStr.SavePerfLog;
var
  i: Integer;
  t: TextFile;
begin
  AssignFile(t, self.ClassName + '.perfL.txt');
  Rewrite(t);
  for i := 0 to PerfLog.Count - 1 do
  begin
    WriteLN(t, PerfLog[i]);
  end;
  CloseFile(t);
end;

procedure TTestPasStr.TearDownOnce;
begin
  SavePerfLog;
  PerfLog.Free;
  inherited TearDownOnce;
end;

{ TTestStringBuilder }

procedure TTestStringBuilder.TestEmptyString;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  CheckEquals('', s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestCreateDestroy;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Free;
  Check(True);
end;

procedure TTestStringBuilder.TestSingleItem;
const
  TestingString = 'abc';
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add(TestingString);
  CheckEquals(TestingString, s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestTwoItems;
const
  TestingString1 = 'abc';
  TestingString2 = 'def';
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add(TestingString1);
  CheckEquals(TestingString1, s.ToString);
  s.Add(TestingString2);
  CheckEquals(TestingString1 + TestingString2, s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestTwoItemsA;
const
  TestingString1 = 'abc';
  TestingString2 = 'def';
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add([TestingString1, TestingString2]);
  CheckEquals(TestingString1 + TestingString2, s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestWithClean;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add('ab');
  s.Add('cd');
  CheckEquals('abcd', s.ToString);
  s.Clean;
  CheckEquals('', s.ToString);
  s.Add('ab');
  s.Add('c');
  s.Add('def');
  CheckEquals('abcdef', s.ToString);
  s.Clean;
  CheckEquals('', s.ToString);
  s.Add('fff');
  CheckEquals('fff', s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestRus;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add('Колбас');
  s.Add('ка');
  CheckEquals('Колбаска', s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestCleanBeforeFree;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add('Meow');
  s.Clean;
  CheckEquals('', s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestSingleChar;
var
  s: TStringBuilder;
begin
  s := TStringBuilder.Create;
  s.Add('A');
  CheckEquals('A', s.ToString);
  s.Free;
end;

procedure TTestStringBuilder.TestSingleChars;
const
  Chars = '12334 asdf asdf asdf';
var
  s: TStringBuilder;
  i: Integer;
begin
  s := TStringBuilder.Create;
  for i := 1 to Length(Chars) do
    s.Add(Chars[i]);
  CheckEquals(Chars, s.ToString);
  s.Free;
end;

{ TTestStringLinkedList }

procedure TTestStringLinkedList.TestEmpty;
var
  list: TStringLinkedList;
procedure go(const s: string);
begin
  Fail('List is empty but go procedure is called');
end;
begin
  list := TStringLinkedList.Create;
  list.ForEach(@go);
  CheckEquals('', list.ToString(''));
  list.Free;
end;

procedure TTestStringLinkedList.TestSingle;
const
  TestingString = 'lal';
var
  list: TStringLinkedList;
  n: Integer;
procedure go(const s: string);
begin
  CheckEquals(TestingString, s);
  n := n + 1;
end;
begin
  list := TStringLinkedList.Create;
  list.Add(TestingString);
  n := 0;
  list.ForEach(@go);
  CheckEquals(1, n);
  CheckEquals(TestingString, list.ToString(''));
  CheckEquals(TestingString, list.ToString('X'));
  CheckEquals(TestingString, list.ToString('YY'));
  list.Free;
end;

procedure TTestStringLinkedList.TestGenerateRandomString;
const
  a = DefaultTestingStringLengthA;
  b = DefaultTestingStringLengthB;
var
  s: string;
  l, i: Integer;
begin
  for i := 1 to 10 do
  begin
    s := GenerateRandomString(a, b);
    l := Length(s);
    Check((a <= l) and (l <= b));
  end;
end;

procedure TTestStringLinkedList.CheckForEach(const a: TStringDynArray; aList: TStringLinkedList);
var
  i: Integer;
procedure go(const s: string);
begin
  CheckEquals(a[i], s);
  i := i + 1;
end;
begin
  i := 0;
  aList.ForEach(@go);
  CheckEquals(i, Length(a));
end;

procedure TTestStringLinkedList.CheckListToString(
  const aExpected: TStringDynArray;
  aList: TStringLinkedList;
  const aSeparator: string
);
var
  expectedS: string;
  actualS: string;
procedure formExpectedString;
begin
  expectedS := JoinStringArray(aExpected, aSeparator);
end;
procedure formActualString;
begin
  actualS := aList.ToString(aSeparator);
end;
begin
  RunPerf(
    @formExpectedString,
    'CheckListToString_NJ(' + IntToStr(Length(aExpected)) + ',' + aSeparator + ')'
  );
  RunPerf(
    @formActualString,
    'CheckListToString_LJ(' + IntToStr(Length(aExpected)) + ',' + aSeparator + ')'
  );
  CheckEquals(expectedS, actualS);
end;

procedure TTestStringLinkedList.TestMultiple;
{%REGION ProcedureHead}
const
  N = DefaultTestingStringArrayLength;
var
  a: TStringDynArray;
  i: Integer;
  list: TStringLinkedList;
{%endregion ProcedureHead}
begin
  list := TStringLinkedList.Create;
  a := GenerateTestingArray(DefaultTestingStringLengthA, DefaultTestingStringLengthB, N);
  for i := 0 to N - 1 do
    list.Add(a[i]);
  CheckForEach(a, list);
  CheckListToString(a, list, '');
  CheckListToString(a, list, 'X');
  CheckListToString(a, list, 'YY');
  list.Free;
end;

{ TTestStringUtil }

procedure TTestStringUtil.TestOpenStringArrayToDynArraySimple;
var
  a: TStringDynArray;
begin
  a := OpenStringArrayToDynArray(['ab', 'cd', 'e']);
  CheckEquals(3, Length(a));
  CheckEquals('ab', a[0]);
  CheckEquals('cd', a[1]);
  CheckEquals('e', a[2]);
end;

procedure TTestStringUtil.TestOpenStringArrayToDynArrayEmpty;
var
  a: TStringDynArray;
begin
  a := OpenStringArrayToDynArray([]);
  CheckEquals(0, Length(a));
end;

procedure TTestStringUtil.TestSplitStringSimple;
var
  a: TStringDynArray;
begin
  a := SplitString('ab;cd;e', ';');
  CheckEquals(3, Length(a));
  CheckEquals('ab', a[0]);
  CheckEquals('cd', a[1]);
  CheckEquals('e', a[2]);
end;

procedure TTestStringUtil.TestSplitStringWithSep(aSeparator: Char);
const
  N = DefaultTestingStringArrayLength;
var
  a, b: TStringDynArray;
  s: string;
  i: Integer;
begin
  a := GenerateTestingArray(0, DefaultTestingStringLengthB, N);
  s := JoinStringArray(a, aSeparator);
  b := SplitString(s, aSeparator);
  for i := 0 to N - 1 do
    CheckEquals(a[i], b[i]);
end;

procedure TTestStringUtil.TestSplitStringMultiple;
begin
  TestSplitStringWithSep(':');
  TestSplitStringWithSep('.');
  TestSplitStringWithSep('!');
end;

procedure TTestStringUtil.TestStringArrayContainsNormal;
var
  a: TStringDynArray;
  s: string;
  i, n: Integer;
  contains: Boolean;
procedure evalContains;
begin
   contains := StringArrayContains(a, s);
end;

begin
  n := DefaultTestingStringArrayLength;
  a :=
    GenerateTestingArray(
      DefaultTestingStringLengthA,
      DefaultTestingStringLengthB,
      n
    )
  ;
  i := Random(n);
  s := a[i];
  evalContains;
  CheckEquals(True, contains);
  s := StringOfChar(NonRandomSymbol, DefaultTestingStringLengthA);
  RunPerf(@evalContains, 'StringArrayContains(' + IntToStr(n) + ',' + IntToStr(i) + ')');
  CheckEquals(False, contains);
end;

procedure TTestStringUtil.TestStringCopyABSimple;
begin
  CheckEquals(
    'mebody ',
    StringCopyAB('Somebody dance with me', 3, 9)
  );
end;

end.

