{ StringUtil unit

  Copyright (C) 2014 by Alexander Savinykh hinstance@yandex.ru

  This library is free software; you can redistribute it and/or modify it under the terms of the
  GNU Library General Public License as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version with the following modification:

  As a special exception, the copyright holders of this library give you permission to link this
  library with independent modules to produce an executable, regardless of the license terms of
  these independent modules,and to copy and distribute the resulting executable under terms of
  your choice, provided that you also meet, for each linked independent module, the terms and
  conditions of the license of that module. An independent module is a module which is not derived
  from or based on this library. If you modify this library, you may extend this exception to your
  version of the library, but you are not obligated to do so. If you do not wish to do so, delete
  this exception statement from your version.

  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
  the GNU Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License along with this
  library; if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
}

unit StringUtilU;

{$Include PasStr.inc}

interface

uses
  types, sysutils, Classes;

function OpenStringArrayToDynArray(const aStrings: array of string): TStringDynArray;
function SplitString(const aText: string; const aSeparator: Char): TStringDynArray;
function StringArrayContains(const a: TStringDynArray; const aWhat: string): Boolean;
function JoinStringArray(const a: TStringDynArray; const aSeparator: string): string;
//< Naive implementation with s := s + a[i]
function StringCopyAB(const s: string; const a, b: Integer): string;
{<
  @longcode(
    StringCopyAB('Somebody dance with me', 3, 9);
    //              3     9
    // returns     'mebody '
  )
}
procedure StringDeleteAB(var s: string; const a, b: Integer);

type

  { TStringsHelper }

  TStringsHelper = class helper for TStrings
    public function GetDelimitedText(const aDelimiter: string): string;
    public function TryGetValue(const aName: string; out aValue: string): Boolean;
  end;

procedure WriteString(a: TStream; const s: string);
function CreateMemoryStream(const s: string): TStream;
function AsString(a: TStream): string;
function AsString(a: Pointer): string;

implementation

function OpenStringArrayToDynArray(const aStrings: array of string): TStringDynArray;
var
  i: Integer;
begin
  SetLength(result, Length(aStrings));
  for i := 0 to Length(aStrings) - 1 do
    result[i] := aStrings[i];
end;

function SplitString(const aText: string; const aSeparator: Char): TStringDynArray;
var
  i: Integer;
  stringList: TStringList;
begin
  stringList := TStringList.Create;
  stringList.Delimiter := aSeparator;
  stringList.StrictDelimiter := True;
  stringList.DelimitedText := aText;
  SetLength(result, stringList.Count);
  for i := 0 to stringList.Count - 1 do
    result[i] := stringList[i];
  stringList.Free;
end;

function StringArrayContains(const a: TStringDynArray; const aWhat: string): Boolean;
var
  i: Integer;
begin
  result := False;
  for i := 0 to Length(a) - 1 do
    if a[i] = aWhat then
    begin
      result := True;
      break;
    end;
end;

function JoinStringArray(const a: TStringDynArray; const aSeparator: string): string;
var
  i, lengthOfA: Integer;
begin
  result := '';
  lengthOfA := Length(a);
  for i := 0 to lengthOfA - 2 do
  begin
    result := result + a[i] + aSeparator;
  end;
  if lengthOfA >= 1 then
    result := result + a[lengthOfA - 1];
end;

function StringCopyAB(const s: string; const a, b: Integer): string;
begin
  result := Copy(s, a, b - a + 1);
end;

procedure StringDeleteAB(var s: string; const a, b: Integer);
begin
  Delete(s, a, b - a + 1);
end;

{ TStringsHelper }

function TStringsHelper.GetDelimitedText(const aDelimiter: string): string;
var
  i, resultLength: Integer;
  s: string;
begin
  {%Region Prepare}
  resultLength := 0;
  for i := 0 to Count - 1 do
  begin
    Inc(resultLength, Length(self[i]));
    if i <> Count - 1 then
      Inc(resultLength, Length(aDelimiter));
  end;
  SetLength(result, resultLength);
  {%EndRegion}
  {%Region Fill}
  resultLength := 0;
  for i := 0 to Count - 1 do
  begin
    s := self[i];
    System.Move(s[1], result[resultLength + 1], Length(s));
    Inc(resultLength, Length(s));
    if i <> Count - 1 then
    begin
      System.Move(aDelimiter[1], result[resultLength + 1], Length(aDelimiter));
      Inc(resultLength, Length(aDelimiter));
    end;
  end;
  {%EndRegion}
end;

function TStringsHelper.TryGetValue(const aName: string; out aValue: string): Boolean;
var
  i: Integer;
begin
  i := IndexOfName(aName);
  result := i >= 0;
  case result of
  True:
    aValue := ValueFromIndex[i];
  False:
    aValue := '';
  end;
end;

procedure WriteString(a: TStream; const s: string);
begin
  a.Write(s[1], Length(s));
end;

function CreateMemoryStream(const s: string): TStream;
begin
  result := TMemoryStream.Create;
  WriteString(result, s);
end;

function AsString(a: TStream): string;
var
  n: Integer;
begin
  a.Position := 0;
  n := a.Size;
  if n < MaxInt then
  begin
    SetLength(result, n);
    a.Read(result[1], n);
  end
  else
    result := '';
end;

function AsString(a: Pointer): string;
begin
  result := IntToHex(UIntPtr(a), SizeOf(a) * 2);
end;


end.

